// BlobBase.h
// Definit la classe blob de base

#ifndef V3D_BLOBS_BLOB_BASE_H_INCLUDED
#define V3D_BLOBS_BLOB_BASE_H_INCLUDED

#include "Basics/MatrixTransf.h"
#include "Basics/Vector.h"

#include "Basics/BoundAabb.h"

namespace V3D{

/////////////////////////////////////////////////////////////////////////////////////////
class BlobBase	
{
public:
	enum EEvalType
	{
		etQuadratic = 0,
		etPolynom4, 
		etPolynom6,
		etLast
	};

	typedef std::vector<int32>   VertexIndexArray;       // Liste des indices de vertex influences par le blob
	typedef std::vector<float>   VertexInfluenceArray;   // Liste des valeurs d'influence sur les vertex par le blob

public:
	BlobBase(float fCenterEnergy, EEvalType etStyle = BlobBase::etPolynom4);
	virtual ~BlobBase() {}

	// Methode de clonage d'un blob. A redefinir par les classes derivees
	virtual BlobBase* Clone() const = 0;


	// Modification de l'energie au centre du blob
	void SetCenterEnergy(float fEnergy);

	// Modification de l'etendue du blob
	void SetInfluenceRange(float fInfl);

	// Changement de style de blob defini par la formule de decroissance du potentiel avec la distance au "centre" du blob
	void SetStyle(EEvalType etStyle);


	// Recuperation de la boite englobante du volume de l'espace ou le blob a un potentiel non nul.
	// retourne vrai si un tel encadrement existe
	bool GetBlobBounds( Vector3D& vcBlobMin, Vector3D& vcBlobMax ) const;
	bool GetBlobBounds( BoundAabb& blobBounds ) const;
	inline const BoundAabb& GetBlobBounds() const;

	const Vector3D& GetBlobBoundsMin() const { return m_boundAABB.GetMin(); }
	const Vector3D& GetBlobBoundsMax() const { return m_boundAABB.GetMax(); }

	// Determine les vertices du tableau aPos influencees (cad pour lesquelles le potentiel est non nul) par le blob
	void FindInfluencedVertices( VertexIndexArray& anVtxInfluencedIdx, const Vect3DArray& aPos) const;


	// Idem pour un sous-tableau
	virtual void FindInfluencedVerticesSubArray( VertexIndexArray& anVtxInfluencedIdx, const Vect3DArray& aPos,
	                                              int32 nStart, int32 nCount) const;


	// Pour chaque vertex du sous-tableau fourni, ajout du gradient associe a ce blob
	// Determine en meme temps les vertices influences (idem a FindInfluencedVerticesSubArray)
	virtual void GradientAddAtSubArrayFillInfluenced( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                                  VertexIndexArray& anInflVtxToFind,
	                                                  int32 nStart, int32 nCount) const = 0;

	// Idem a partir du sous-ensemble de vertices dont on sait qu'ils sont influences par le blob
	virtual void GradientAddAtInfluencedVertices( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                              const VertexIndexArray& anInflVtx) const = 0;

	// Ajout du gradient pour une position unique
	// ATT : Lent, il vaut mieux utiliser les methodes precedentes
	virtual bool GradientAddAtPos( Vector3f& vcGrad, const Vector3D& pos) const = 0;


	// Ajout des potentiels de tous les points de la grille fournie pour le blob
	virtual void EnergyAddAtGrid( float* pafAddEnergies, 
						const Vector3D& vcStart, const Vector3D& vcSteps,
						int32 nStepsX, int32 nStepsY, int32 nStepsZ) const = 0;


	virtual void EnergyGetAtInfluencedVertices( VertexInfluenceArray& afVals, const Vect3DArray& aPos, 
                                                const VertexIndexArray& anVertexIdx) const = 0;


	virtual bool EnergyAddAtPos( float& fDest, const Vector3D& pos) const = 0;

	// Calcule a partir de la boite definie par vcBoxMin et vcBoxMax, son sous-ensemble minimum contenant le blob
	// Retourne true si cette boite minimum existe (volume non nul)
	virtual bool ComputeMinEnclosingBoxAligned( Vector3D& vcMin, Vector3D& vcMax, 
												const Vector3D& vcBoxMin, const Vector3D& vcBoxMax) const;


protected:

	virtual bool ComputeBlobBounds( Vector3D& vcBlobMin, Vector3D& vcBlobMax) const;


	inline EEvalType GetStyle() const;

	void NotifyBoundsChange();

	inline float EnergyNotZeroAtSqrDist( const float& fSqrDist) const;

	// Pour la distance au carre du point au blob, calcule l'energie.
	// Specialisation par style de decroissance, pour des performances optimales
	template <int EVALTYPE>
		inline float EnergyNotZeroGetAtSqrDistConfig( const float& fSqrDist) const;

	// Pour la distance au carre du point au blob, calcule le facteur multiplicateur du gradient.
	// Specialisation par style de decroissance, pour des performances optimales
	template <int EVALTYPE>
		inline float GradientGetVectorFactorNotZeroConfig( const float& fSqrDist) const;

	// Factorisation de la definition de la methode Clone
	template <class BLOBCLASS>
		static BLOBCLASS* CloneClass(const BLOBCLASS& that) { return new BLOBCLASS(that); }


#define BLOBBASE_STD_OVERRIDES( BLOBCLASS )  \
   virtual BlobBase* Clone() const           \
   {                                         \
     return CloneClass(*this);               \
   }



private:
	EEvalType   m_etEvaluation;                 // Type d'evaluation (type de decroissance d'energie)

	float       m_fFullEnergy;                  // Energie au centre du blob

	bool        m_bBounded;
	BoundAabb m_boundAABB;
};


//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////

inline BlobBase::EEvalType BlobBase::GetStyle() const
{
	return m_etEvaluation;
}

inline const BoundAabb& BlobBase::GetBlobBounds() const
{
	return m_boundAABB;
}




// Retourne l'energie generee par la boule a la distance dont le carre est fSqrDist
template <int EVALTYPE>
inline float BlobBase::EnergyNotZeroGetAtSqrDistConfig( const float& fSqrDist) const
{
	assert( fSqrDist <= 1.f + 1E-5f );

	switch (EVALTYPE)
	{
	case etQuadratic:
		{
			float fRadDivInflSqr = fSqrDist;
			if( fRadDivInflSqr < (1.f / 9.f) )
			{
				// Rayon < rayon de seuil / 3 ==> energie = formule 1
				return m_fFullEnergy * ( 1.f - 3.f * fRadDivInflSqr);
			}
			else 
			{
				// Rayon entre rayon de seuil / 3 et rayon de seuil ==> energie = formule 2
				float fTmpRadDivInfl = 1.f - Sqrt(fRadDivInflSqr);
				return  m_fFullEnergy * (3.f / 2.f) * fTmpRadDivInfl * fTmpRadDivInfl;
			}
		}
		break;

	case etPolynom4:
		{
			float fTmp = 1.f - fSqrDist;
			return m_fFullEnergy * fTmp * fTmp;
		}
		break;

	case etPolynom6:
		{
			return m_fFullEnergy * (1.f + fSqrDist * ( (-22.f/9.f) + fSqrDist * ((17.f/9.f) - fSqrDist * (4.f/9.f))));
		}
		break;

	default:
		assert(0);
		return 0.f;
		break;
	}
}

								///////////////////////////////

template<int EVALTYPE>
inline float BlobBase::GradientGetVectorFactorNotZeroConfig( const float& fSqrDist) const
{
	// Rayon >= rayon de seuil ==> energie = 0, gradient inchange
	float fFactor;                                         
	switch (EVALTYPE)
	{
	case etQuadratic:
		{
			if( fSqrDist > (1.f / 9.f) )
			{
				float fSqrDistUsed = fSqrDist;
				// Rayon entre rayon de seuil / 3 et rayon de seuil ==> energie = formule 2
				float fFactorTmp = 3.f * m_fFullEnergy; // * m_fInfluenceRangeInv; 
				//		fFactor = fFactorTmp * float( ( 1.0 / (sqrt( fSqrDist))) - m_fInfluenceRangeInv );
				fFactor = fFactorTmp * float( ( 1.0 / (sqrt(fSqrDistUsed))) - 1.0 );
			}
			else
			{
				// Rayon < rayon de seuil / 3 ==> energie = formule 1
				fFactor = 6.f * m_fFullEnergy; //  * m_fInfluenceRangeInvSqr;
			}
		}
		break;

	case etPolynom4:
		// Calcul du gradient : Grad = 4C/R *(1 - r/R) * vcOffs (R est Influence Range = 1)
		fFactor = 4.f * m_fFullEnergy * (1.f - fSqrDist);
		break;

	case etPolynom6:
		fFactor = - m_fFullEnergy * ((-44.f/9.f) + fSqrDist * (68.f/9.f - fSqrDist * (24.f/9.f)));
		break;

	default:
		fFactor = 0.f;
		assert(0);
		break;
	}
	return fFactor;
}





inline float BlobBase::EnergyNotZeroAtSqrDist( const float& fSqrDist) const
{
	float fFactor;
	switch( m_etEvaluation )
	{
	case etQuadratic : fFactor = EnergyNotZeroGetAtSqrDistConfig<int(etQuadratic) > (fSqrDist); break;
	case etPolynom4  : fFactor = EnergyNotZeroGetAtSqrDistConfig<int(etPolynom4)  > (fSqrDist); break;
	case etPolynom6  : fFactor = EnergyNotZeroGetAtSqrDistConfig<int(etPolynom6)  > (fSqrDist); break;
	default: 
		assert(0);
		fFactor = 0.f;
		break;
	}
	return fFactor;
}



} // namespace


#endif	// #ifndef V3D_BLOBS_BLOB_BASE_H_INCLUDED

